/*
 * Copyright (c) 2009, Felix Opatz
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of  nor the names of its contributors may be used to
 *    endorse or promote products derived from this software without specific
 *    prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string>
#include <vector>
#include <sstream>
#include <windows.h>
#include <commctrl.h>

#include "ImportList.h"

// Width, height and name of main window
#define APP_WIDTH        640
#define APP_HEIGHT       480
#define APP_NAME         "ImportDemoGui"

// States for undoing clipboard activities
#define CLIPBOARD_OPENED 1
#define MEMORY_ALLOCATED 2
#define MEMORY_LOCKED    3

// Scancode for 'c' key
#define SCANCODE_C       46

// Files dropped into window
std::vector<std::string> DroppedFiles;
// Original window proc of tree view control
WNDPROC pfOrigWindowProc;

// Splits the given symbol into parts delimited by Delimiter
int SplitSymbol(std::string &Symbol, const std::string &Delimiter, std::vector<std::string> &Parts)
{
    size_t pos = 0;
    size_t oldpos;

    // Remove previous content
    Parts.clear();

    do
    {
        // Remember last occurence
        oldpos = pos;
        // Find next occurence
        pos = Symbol.find(Delimiter, oldpos);
        // Insert substring into result vector
        Parts.push_back(Symbol.substr(oldpos, pos));
        // Skip delimiter for next turn, if there is a next turn
        if (pos < std::string::npos)
            pos +=  Delimiter.length();
    }
    while (pos != std::string::npos);

    // Return number of parts
    return Parts.size();
}

// Inserts the symbols as children of hParent; called recursively
int InsertSymbols(HWND hWnd, HTREEITEM hParent, std::vector<std::string> &Symbols,
                  int iStartPos, std::string ParentPart, const std::string &Delimiter, int iLevel)
{
    std::vector<std::string> SymbolParts;
    TV_INSERTSTRUCT isNewItem;
    int iPos;
    int iCount;
    std::string CurrentPart;

    // Remember our original parent
    isNewItem.hParent = hParent;
    isNewItem.hInsertAfter = TVI_LAST;
    isNewItem.item.mask = TVIF_TEXT;

    // Loop until end of symbol list
    for (iPos = iStartPos; iPos < (int)Symbols.size(); iPos++)
    {
        // Split the symbol into parts (module name, function name)
        iCount = SplitSymbol(Symbols[iPos], Delimiter, SymbolParts);

        // There must be enough parts for this recursion level
        if (iCount > iLevel)
        {
            // If we are not at the top level and the parent's symbol part has
            // changed (e.g. next module) we need to ascend
            if ((iLevel > 0) && (SymbolParts[iLevel - 1] != ParentPart))
                break;

            // If the current part has not been inserted, insert it now
            if (SymbolParts[iLevel] != CurrentPart)
            {
                // Remember current part for next loop
                CurrentPart = SymbolParts[iLevel].c_str();
                // Insert current item as new item
                isNewItem.item.pszText = (CHAR*)CurrentPart.c_str();
                isNewItem.item.cchTextMax = CurrentPart.length();
                // Remember this item as new parent for next child nodes
                hParent = TreeView_InsertItem(hWnd, &isNewItem);
            }

            // Insert next level of symbols; skip number of items inserted by next level
            iPos += InsertSymbols(hWnd, hParent, Symbols, iStartPos + iPos, CurrentPart, Delimiter, iLevel + 1);
        }
        else
            break;
    }

    // Return number of inserted items
    return iPos - iStartPos;
}

// Updates the result window, hWnd is the tree view's handle
void UpdateResult(HWND hWnd)
{
    std::vector<std::string>::iterator it;
    std::vector<std::string> ImportedSymbols;
    std::string ErrorMsg;
    TV_INSERTSTRUCT isNewItem;
    HTREEITEM hParent;

    // Remove all items for new result
    TreeView_DeleteAllItems(hWnd);

    // Process all dropped files
    for (it = DroppedFiles.begin(); it != DroppedFiles.end(); it++)
    {
        // New children of root
        isNewItem.hParent = TVI_ROOT;
        isNewItem.hInsertAfter = TVI_LAST;
        isNewItem.item.mask = TVIF_TEXT;

        // Try to get list for current file
        if (ImportList::GetList(*it, ImportedSymbols, ImportList::LT_FUNCTIONS))
        {
            // Insert filename
            isNewItem.item.pszText = (CHAR*)it->c_str();
            isNewItem.item.cchTextMax = it->length();
            hParent = TreeView_InsertItem(hWnd, &isNewItem);

            // Insert symbols as children
            InsertSymbols(hWnd, hParent, ImportedSymbols, 0, "", ImportList::ListDelimiter, 0);
        }
        else
        {
            // Failed, insert error message instead
            ErrorMsg = "Could not read " + *it;
            isNewItem.item.pszText = (CHAR*)ErrorMsg.c_str();
            isNewItem.item.cchTextMax = ErrorMsg.length();
            TreeView_InsertItem(hWnd, &isNewItem);
        }
    }
}

// Called when files have been dropped
void OnDropFiles(HWND hWnd, HDROP hDrop)
{
    UINT uFileCount;
    UINT uFileIndex;
    CHAR acFilePath[MAX_PATH];

    // Query count of dropped files
    uFileCount = DragQueryFile(hDrop, 0xFFFFFFFF, NULL, 0);

    // Clear vector of previously dropped files
    DroppedFiles.clear();

    // Insert dropped files into vector
    for (uFileIndex = 0; uFileIndex < uFileCount; uFileIndex++)
    {
        DragQueryFile(hDrop, uFileIndex, acFilePath, sizeof(acFilePath));
        DroppedFiles.push_back(acFilePath);
    }

    // Update the result
    UpdateResult(hWnd);
}

// Generates a string with the tree data
int GetTreeData(HWND hWnd, std::string &TreeData, HTREEITEM hItem, int iLevel)
{
    TVITEM tvItem;
    CHAR acItemText[MAX_PATH];
    int iCount;
    int iLines = 0;

    // Loop while hItem is a valid tree item
    while (hItem != NULL)
    {
        // Query the text entry of this item
        tvItem.hItem = hItem;
        tvItem.mask = TVIF_TEXT;
        tvItem.pszText = acItemText;
        tvItem.cchTextMax = sizeof(acItemText);
        TreeView_GetItem(hWnd, &tvItem);

        // Indent for lower levels
        for (iCount = 0; iCount < iLevel; iCount++)
            TreeData = TreeData + "\t";

        // Insert data of current item into string
        TreeData = TreeData + acItemText;
        TreeData = TreeData + "\r\n";
        iLines++;

        // Descend to next level and add its lines
        iLines += GetTreeData(hWnd, TreeData, TreeView_GetChild(hWnd, hItem), iLevel + 1);

        // Continue with next sibling
        hItem = TreeView_GetNextSibling(hWnd, hItem);

        // If at top level and not the last item, insert a newline
        if ((iLevel == 0) && (hItem != NULL))
        {
            TreeData = TreeData + "\r\n";
            iLines++;
        }
    }

    // Return number of lines for this level and all levels below
    return iLines;
}

// Copies the tree to the clipboard
void CopyTreeToClipboard(HWND hWnd)
{
    std::string TreeData;
    HGLOBAL hMem = NULL;
    char *pMem;
    int iLines;
    std::stringstream sstream;
    int iState = 0;

    // Get tree data into std::string
    iLines = GetTreeData(hWnd, TreeData, TreeView_GetRoot(hWnd), 0);

    // If nothing to do, don't destroy clipboard data
    if (iLines == 0)
        return;

    try
    {
        // Open clipboard
        if (OpenClipboard(hWnd) == FALSE)
            throw GetLastError();
        iState = CLIPBOARD_OPENED;

        // Remove current content
        if (EmptyClipboard() == FALSE)
            throw GetLastError();

        // Get memory for clipboard data
        hMem = GlobalAlloc(GMEM_MOVEABLE, TreeData.length() + 1);
        if (hMem == NULL)
            throw GetLastError();
        iState = MEMORY_ALLOCATED;

        // Get pointer to memory
        pMem = (char*)GlobalLock(hMem);
        if (pMem == NULL)
            throw GetLastError();
        iState = MEMORY_LOCKED;

        // Copy the string into memory
        strcpy(pMem, TreeData.c_str());

        // Unlock memory
        if (GlobalUnlock(hMem) != FALSE)
            throw GetLastError();
        iState = MEMORY_ALLOCATED;

        // Set clipboard data to memory
        if (SetClipboardData(CF_TEXT, hMem) == NULL)
            throw GetLastError();
        iState = CLIPBOARD_OPENED;

        // Close clipboard
        if (CloseClipboard() == FALSE)
            throw GetLastError();
    }
    catch (DWORD dwErrorCode)
    {
        // Construct error message and display it
        sstream << "Caught exception with error code " << dwErrorCode;
        MessageBox(hWnd, sstream.str().c_str(), APP_NAME, MB_ICONERROR);

        // Free allocated resources
        switch (iState)
        {
            case MEMORY_LOCKED:
                GlobalUnlock(hMem);
            case MEMORY_ALLOCATED:
                GlobalFree(hMem);
            case CLIPBOARD_OPENED:
                CloseClipboard();
        }
    }
}

// WindowProc, all messages go through this function first
LRESULT CALLBACK WindowProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        // Close application
        case WM_CLOSE:
            PostQuitMessage(0);
            return 0;

        // Files have been dropped
        case WM_DROPFILES:
            OnDropFiles(hWnd, (HDROP)wParam);
            return 0;

        // CTRL+C was pressed
        case WM_CHAR:
            if ((LOBYTE(HIWORD(lParam)) == SCANCODE_C) && (GetKeyState(VK_LCONTROL) < 0))
                CopyTreeToClipboard(hWnd);
            return 0;
    }

    // Call the original WindowProc to get its functionality
    return CallWindowProc(pfOrigWindowProc, hWnd, uMsg, wParam, lParam);
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;
    HWND hWnd;

    // Initialize the Common Controls
    InitCommonControls();

    // Create the TreeView control
    hWnd = CreateWindowEx(WS_EX_ACCEPTFILES,    // Ex Styles
                          WC_TREEVIEW,          // Window class
                          APP_NAME,             // Window name
                          WS_VISIBLE | WS_OVERLAPPEDWINDOW | TVS_HASLINES | 
                          TVS_HASBUTTONS | TVS_LINESATROOT | TVS_SHOWSELALWAYS,   // Styles
                          CW_USEDEFAULT,        // x postion
                          CW_USEDEFAULT,        // y position
                          APP_WIDTH,            // width
                          APP_HEIGHT,           // height
                          GetDesktopWindow(),   // parent
                          NULL,                 // ID
                          hInstance,            // Instance
                          NULL);                // Param, unused

    // Install our own window proc to add some functionality
    pfOrigWindowProc = (WNDPROC)SetWindowLong(hWnd, GWL_WNDPROC, (LONG)WindowProc);

    // Message loop
    while (GetMessage(&msg, NULL, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    // Return code of WM_QUIT
    return msg.wParam;
}
